<?php
class FlussonicAPI {
    private $serverUrl;
    private $apiKey;
    
    public function __construct($serverId = null) {
        if ($serverId) {
            $db = new Database();
            $stmt = $db->query("SELECT url, api_key FROM flussonic_servers WHERE id = ?", [$serverId]);
            $server = $stmt->fetch();
            
            if ($server) {
                $this->serverUrl = rtrim($server['url'], '/');
                $this->apiKey = $server['api_key'];
            }
        }
    }
    
    private function callAPI($endpoint, $method = 'GET', $data = null) {
        $url = $this->serverUrl . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, FLUSSONIC_API_TIMEOUT);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->apiKey
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method === 'PUT') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method === 'DELETE') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        return [
            'status' => $httpCode,
            'data' => json_decode($response, true)
        ];
    }
    
    // Stream Management
    public function createStream($streamName, $inputUrl, $options = []) {
        $endpoint = '/flussonic/api/streamer';
        $data = [
            'url' => $inputUrl,
            'dynamic' => $options['dynamic'] ?? false,
            'autostart' => $options['autostart'] ?? true,
            'reconnect' => $options['reconnect'] ?? true,
            'buffertime' => $options['buffertime'] ?? 2000
        ];
        
        return $this->callAPI($endpoint, 'POST', $data);
    }
    
    public function getStreams() {
        $endpoint = '/flussonic/api/streams';
        return $this->callAPI($endpoint);
    }
    
    public function getStreamInfo($streamName) {
        $endpoint = '/flussonic/api/stream/' . $streamName;
        return $this->callAPI($endpoint);
    }
    
    public function updateStream($streamName, $data) {
        $endpoint = '/flussonic/api/stream/' . $streamName;
        return $this->callAPI($endpoint, 'PUT', $data);
    }
    
    public function deleteStream($streamName) {
        $endpoint = '/flussonic/api/stream/' . $streamName;
        return $this->callAPI($endpoint, 'DELETE');
    }
    
    // Transcoding
    public function createTranscodingProfile($streamName, $profileName, $settings) {
        $endpoint = "/flussonic/api/stream/{$streamName}/transcoder/{$profileName}";
        return $this->callAPI($endpoint, 'PUT', $settings);
    }
    
    public function getTranscodingProfiles($streamName) {
        $endpoint = "/flussonic/api/stream/{$streamName}/transcoder";
        return $this->callAPI($endpoint);
    }
    
    // Statistics
    public function getStreamStats($streamName) {
        $endpoint = "/flussonic/api/stream/{$streamName}/stats";
        return $this->callAPI($endpoint);
    }
    
    public function getServerStats() {
        $endpoint = '/flussonic/api/server/stats';
        return $this->callAPI($endpoint);
    }
    
    // Check if server is online
    public function ping() {
        $endpoint = '/flussonic/api/ping';
        $result = $this->callAPI($endpoint);
        return $result['status'] === 200;
    }
}