<?php
class Auth {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
        session_start();
    }
    
    // Admin authentication
    public function adminLogin($username, $password) {
        $stmt = $this->db->query(
            "SELECT id, username, email, password, role FROM admins WHERE username = ? AND status = 1",
            [$username]
        );
        
        $admin = $stmt->fetch();
        
        if ($admin && password_verify($password, $admin['password'])) {
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_role'] = $admin['role'];
            $_SESSION['admin_login_time'] = time();
            
            // Update last login
            $this->db->query(
                "UPDATE admins SET last_login = NOW() WHERE id = ?",
                [$admin['id']]
            );
            
            // Log activity
            $this->logAdminActivity($admin['id'], 'login', 'Admin logged in');
            
            return true;
        }
        
        return false;
    }
    
    public function adminLogout() {
        if (isset($_SESSION['admin_id'])) {
            $this->logAdminActivity($_SESSION['admin_id'], 'logout', 'Admin logged out');
        }
        
        session_destroy();
    }
    
    public function isAdminLoggedIn() {
        if (!isset($_SESSION['admin_id'])) {
            return false;
        }
        
        // Check session timeout
        if (time() - $_SESSION['admin_login_time'] > SESSION_TIMEOUT) {
            $this->adminLogout();
            return false;
        }
        
        return true;
    }
    
    // User authentication
    public function userLogin($username, $password) {
        $stmt = $this->db->query(
            "SELECT id, username, email, password, package_id, is_active FROM users WHERE username = ?",
            [$username]
        );
        
        $user = $stmt->fetch();
        
        if ($user && password_verify($password, $user['password']) && $user['is_active']) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_username'] = $user['username'];
            $_SESSION['user_package'] = $user['package_id'];
            $_SESSION['user_login_time'] = time();
            
            // Generate API token
            $token = bin2hex(random_bytes(32));
            $tokenExpiry = date('Y-m-d H:i:s', strtotime('+30 days'));
            
            $this->db->query(
                "UPDATE users SET last_login = NOW(), token = ?, token_expiry = ? WHERE id = ?",
                [$token, $tokenExpiry, $user['id']]
            );
            
            return ['success' => true, 'token' => $token];
        }
        
        return ['success' => false];
    }
    
    public function userRegister($data) {
        // Check if username or email exists
        $stmt = $this->db->query(
            "SELECT id FROM users WHERE username = ? OR email = ?",
            [$data['username'], $data['email']]
        );
        
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'Username or email already exists'];
        }
        
        // Get package details
        $stmt = $this->db->query(
            "SELECT max_streams, max_bandwidth FROM packages WHERE id = ?",
            [$data['package_id']]
        );
        $package = $stmt->fetch();
        
        // Get available server
        $stmt = $this->db->query(
            "SELECT id FROM flussonic_servers WHERE status = 'active' ORDER BY used_bandwidth ASC LIMIT 1"
        );
        $server = $stmt->fetch();
        
        if (!$server) {
            return ['success' => false, 'message' => 'No available servers'];
        }
        
        // Insert user
        $stmt = $this->db->query(
            "INSERT INTO users (username, email, password, package_id, server_id, max_streams, max_bandwidth) 
             VALUES (?, ?, ?, ?, ?, ?, ?)",
            [
                $data['username'],
                $data['email'],
                password_hash($data['password'], PASSWORD_DEFAULT),
                $data['package_id'],
                $server['id'],
                $package['max_streams'],
                $package['max_bandwidth'] * 1000000 // Convert Mbps to bps
            ]
        );
        
        return ['success' => true, 'user_id' => $this->db->getConnection()->lastInsertId()];
    }
    
    public function isUserLoggedIn() {
        if (!isset($_SESSION['user_id'])) {
            return false;
        }
        
        // Check session timeout
        if (time() - $_SESSION['user_login_time'] > SESSION_TIMEOUT) {
            $this->userLogout();
            return false;
        }
        
        return true;
    }
    
    public function userLogout() {
        session_destroy();
    }
    
    // Token authentication for API
    public function authenticateToken($token) {
        $stmt = $this->db->query(
            "SELECT id, username, package_id FROM users WHERE token = ? AND token_expiry > NOW() AND is_active = 1",
            [$token]
        );
        
        $user = $stmt->fetch();
        
        if ($user) {
            return $user;
        }
        
        return false;
    }
    
    private function logAdminActivity($adminId, $action, $details) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $this->db->query(
            "INSERT INTO admin_logs (admin_id, action, details, ip_address) VALUES (?, ?, ?, ?)",
            [$adminId, $action, $details, $ip]
        );
    }
}