<?php
header('Content-Type: application/json');
require_once '../../includes/Auth.php';
require_once '../../includes/Database.php';
require_once '../../includes/FlussonicAPI.php';

$auth = new Auth();
$db = new Database();

// Check authentication via token or session
$token = $_GET['token'] ?? $_POST['token'] ?? '';
$user = $auth->authenticateToken($token);

if (!$user && !isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

$userId = $user ? $user['id'] : $_SESSION['user_id'];

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'create':
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit;
        }
        
        // Check stream limit
        $stmt = $db->query(
            "SELECT COUNT(*) as count, u.max_streams 
             FROM user_streams us 
             JOIN users u ON us.user_id = u.id 
             WHERE u.id = ? AND us.status = 'active'",
            [$userId]
        );
        $result = $stmt->fetch();
        
        if ($result['count'] >= $result['max_streams']) {
            http_response_code(400);
            echo json_encode(['error' => 'Stream limit reached']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        
        // Get user's server
        $stmt = $db->query("SELECT server_id FROM users WHERE id = ?", [$userId]);
        $userData = $stmt->fetch();
        
        // Create stream on Flussonic server
        $flussonic = new FlussonicAPI($userData['server_id']);
        $streamName = 'user_' . $userId . '_' . time();
        
        $flussonicData = [
            'url' => $data['input_url'],
            'dynamic' => $data['input_type'] === 'pull',
            'autostart' => true
        ];
        
        $result = $flussonic->createStream($streamName, $data['input_url'], $flussonicData);
        
        if ($result['status'] === 200) {
            // Save to database
            $stmt = $db->query(
                "INSERT INTO user_streams (user_id, server_id, stream_name, stream_url, input_type, input_url, output_url) 
                 VALUES (?, ?, ?, ?, ?, ?, ?)",
                [
                    $userId,
                    $userData['server_id'],
                    $streamName,
                    $data['input_url'],
                    $data['input_type'],
                    $data['input_url'],
                    $flussonic->serverUrl . '/' . $streamName . '/index.m3u8'
                ]
            );
            
            $streamId = $db->getConnection()->lastInsertId();
            echo json_encode([
                'success' => true,
                'stream_id' => $streamId,
                'stream_name' => $streamName,
                'output_url' => $flussonic->serverUrl . '/' . $streamName . '/index.m3u8'
            ]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to create stream on Flussonic server']);
        }
        break;
        
    case 'list':
        $stmt = $db->query(
            "SELECT id, stream_name, input_url, output_url, status, current_viewers, current_bandwidth, created_at 
             FROM user_streams 
             WHERE user_id = ? 
             ORDER BY created_at DESC",
            [$userId]
        );
        $streams = $stmt->fetchAll();
        echo json_encode(['success' => true, 'streams' => $streams]);
        break;
        
    case 'stats':
        $streamId = $_GET['stream_id'] ?? 0;
        $stmt = $db->query(
            "SELECT bandwidth, viewers, timestamp 
             FROM stream_statistics 
             WHERE stream_id = ? 
             ORDER BY timestamp DESC 
             LIMIT 100",
            [$streamId]
        );
        $stats = $stmt->fetchAll();
        echo json_encode(['success' => true, 'statistics' => $stats]);
        break;
        
    case 'transcode':
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            exit;
        }
        
        // Check if user has transcoding enabled
        $stmt = $db->query(
            "SELECT p.allow_transcoding, p.max_transcoding_profiles 
             FROM users u 
             JOIN packages p ON u.package_id = p.id 
             WHERE u.id = ?",
            [$userId]
        );
        $package = $stmt->fetch();
        
        if (!$package['allow_transcoding']) {
            http_response_code(403);
            echo json_encode(['error' => 'Transcoding not allowed in your package']);
            exit;
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $streamId = $data['stream_id'];
        $profileName = $data['profile_name'];
        $settings = $data['settings'];
        
        // Get stream details
        $stmt = $db->query(
            "SELECT us.stream_name, u.server_id 
             FROM user_streams us 
             JOIN users u ON us.user_id = u.id 
             WHERE us.id = ? AND us.user_id = ?",
            [$streamId, $userId]
        );
        $stream = $stmt->fetch();
        
        if (!$stream) {
            http_response_code(404);
            echo json_encode(['error' => 'Stream not found']);
            exit;
        }
        
        // Check transcoding profile limit
        $stmt = $db->query(
            "SELECT COUNT(*) as count FROM user_streams 
             WHERE user_id = ? AND JSON_LENGTH(transcoding_profiles) > 0",
            [$userId]
        );
        $result = $stmt->fetch();
        
        if ($result['count'] >= $package['max_transcoding_profiles']) {
            http_response_code(400);
            echo json_encode(['error' => 'Transcoding profile limit reached']);
            exit;
        }
        
        // Create transcoding profile on Flussonic
        $flussonic = new FlussonicAPI($stream['server_id']);
        $result = $flussonic->createTranscodingProfile($stream['stream_name'], $profileName, $settings);
        
        if ($result['status'] === 200) {
            // Update database
            $currentProfiles = $db->query(
                "SELECT transcoding_profiles FROM user_streams WHERE id = ?",
                [$streamId]
            )->fetch()['transcoding_profiles'] ?? '[]';
            
            $profiles = json_decode($currentProfiles, true);
            $profiles[] = [
                'name' => $profileName,
                'settings' => $settings,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $db->query(
                "UPDATE user_streams SET transcoding_profiles = ?, is_transcoding = TRUE WHERE id = ?",
                [json_encode($profiles), $streamId]
            );
            
            echo json_encode(['success' => true, 'profile' => $profiles]);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Failed to create transcoding profile']);
        }
        break;
        
    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid action']);
        break;
}